<?php
if (!defined('ABSPATH')) {
    exit;
}

class Volunpress_Volunteer {
    
    public static function get_all($args = array()) {
        global $wpdb;
        $table = Volunpress_DB::get_table('volunteers');
        
        $defaults = array(
            'is_active' => null,
            'is_committee_member' => null,
            'orderby' => 'total_points',
            'order' => 'DESC',
            'limit' => -1,
            'offset' => 0,
        );
        $args = wp_parse_args($args, $defaults);
        
        $where = array('1=1');
        $values = array();
        
        if ($args['is_active'] !== null) {
            $where[] = 'is_active = %d';
            $values[] = (int) $args['is_active'];
        }
        
        if ($args['is_committee_member'] !== null) {
            $where[] = 'is_committee_member = %d';
            $values[] = (int) $args['is_committee_member'];
        }
        
        $allowed_orderby = array('id', 'first_name', 'last_name', 'total_points', 'joined_date', 'level');
        $orderby = in_array($args['orderby'], $allowed_orderby) ? $args['orderby'] : 'total_points';
        $order = strtoupper($args['order']) === 'ASC' ? 'ASC' : 'DESC';
        
        $sql = "SELECT * FROM $table WHERE " . implode(' AND ', $where) . " ORDER BY $orderby $order";
        
        if ($args['limit'] > 0) {
            $sql .= $wpdb->prepare(" LIMIT %d OFFSET %d", $args['limit'], $args['offset']);
        }
        
        if (!empty($values)) {
            $sql = $wpdb->prepare($sql, $values);
        }
        
        return $wpdb->get_results($sql);
    }
    
    public static function get($id) {
        global $wpdb;
        $table = Volunpress_DB::get_table('volunteers');
        return $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $id));
    }
    
    public static function get_by_email($email) {
        global $wpdb;
        $table = Volunpress_DB::get_table('volunteers');
        return $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE email = %s", $email));
    }
    
    public static function create($data) {
        global $wpdb;
        $table = Volunpress_DB::get_table('volunteers');
        
        $insert_data = array(
            'first_name' => sanitize_text_field($data['first_name']),
            'last_name' => sanitize_text_field($data['last_name']),
            'email' => sanitize_email($data['email']),
            'phone' => isset($data['phone']) ? sanitize_text_field($data['phone']) : '',
            'is_committee_member' => isset($data['is_committee_member']) ? (int) $data['is_committee_member'] : 0,
            'is_active' => isset($data['is_active']) ? (int) $data['is_active'] : 1,
            'membership_paid' => isset($data['membership_paid']) ? (int) $data['membership_paid'] : 0,
            'membership_year' => isset($data['membership_year']) ? (int) $data['membership_year'] : null,
            'joined_date' => isset($data['joined_date']) ? sanitize_text_field($data['joined_date']) : current_time('Y-m-d'),
            'notes' => isset($data['notes']) ? sanitize_textarea_field($data['notes']) : '',
        );
        
        $result = $wpdb->insert($table, $insert_data);
        
        if ($result) {
            $volunteer_id = $wpdb->insert_id;
            
            // Ajouter les points de cotisation si payée
            if (!empty($data['membership_paid'])) {
                self::add_membership_points($volunteer_id);
            }
            
            return $volunteer_id;
        }
        
        return false;
    }
    
    public static function update($id, $data) {
        global $wpdb;
        $table = Volunpress_DB::get_table('volunteers');
        
        $volunteer = self::get($id);
        if (!$volunteer) return false;
        
        $update_data = array();
        $allowed_fields = array('first_name', 'last_name', 'email', 'phone', 'is_committee_member', 'is_active', 'membership_paid', 'membership_year', 'joined_date', 'left_date', 'notes', 'avatar_url');
        
        foreach ($allowed_fields as $field) {
            if (isset($data[$field])) {
                if (in_array($field, array('first_name', 'last_name', 'phone', 'joined_date', 'left_date', 'notes', 'avatar_url'))) {
                    $update_data[$field] = sanitize_text_field($data[$field]);
                } elseif ($field === 'email') {
                    $update_data[$field] = sanitize_email($data[$field]);
                } else {
                    $update_data[$field] = (int) $data[$field];
                }
            }
        }
        
        // Gérer les points de cotisation
        if (isset($data['membership_paid']) && $data['membership_paid'] && !$volunteer->membership_paid) {
            self::add_membership_points($id);
        }
        
        if (empty($update_data)) return true;
        
        return $wpdb->update($table, $update_data, array('id' => $id));
    }
    
    public static function delete($id) {
        global $wpdb;
        $table = Volunpress_DB::get_table('volunteers');
        $activities_table = Volunpress_DB::get_table('activities');
        $claims_table = Volunpress_DB::get_table('claims');
        
        $wpdb->delete($activities_table, array('volunteer_id' => $id));
        $wpdb->delete($claims_table, array('volunteer_id' => $id));
        
        return $wpdb->delete($table, array('id' => $id));
    }
    
    public static function add_points($id, $points) {
        global $wpdb;
        $table = Volunpress_DB::get_table('volunteers');
        
        $wpdb->query($wpdb->prepare(
            "UPDATE $table SET total_points = total_points + %d WHERE id = %d",
            $points, $id
        ));
        
        self::update_level($id);
    }
    
    public static function deduct_points($id, $points) {
        global $wpdb;
        $table = Volunpress_DB::get_table('volunteers');
        
        $wpdb->query($wpdb->prepare(
            "UPDATE $table SET total_points = GREATEST(0, total_points - %d) WHERE id = %d",
            $points, $id
        ));
        
        self::update_level($id);
    }
    
    public static function recalculate_points($id) {
        global $wpdb;
        $table = Volunpress_DB::get_table('volunteers');
        $activities_table = Volunpress_DB::get_table('activities');
        $claims_table = Volunpress_DB::get_table('claims');
        
        $earned = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COALESCE(SUM(points), 0) FROM $activities_table WHERE volunteer_id = %d",
            $id
        ));
        
        $spent = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COALESCE(SUM(points_spent), 0) FROM $claims_table WHERE volunteer_id = %d AND status = 'approved'",
            $id
        ));
        
        $total = max(0, $earned - $spent);
        
        $wpdb->update($table, array('total_points' => $total), array('id' => $id));
        self::update_level($id);
        
        return $total;
    }
    
    public static function update_level($id) {
        global $wpdb;
        $table = Volunpress_DB::get_table('volunteers');
        $levels_table = Volunpress_DB::get_table('levels');
        
        $volunteer = self::get($id);
        if (!$volunteer) return;
        
        $level = $wpdb->get_row($wpdb->prepare(
            "SELECT level_number FROM $levels_table WHERE min_points <= %d ORDER BY min_points DESC LIMIT 1",
            $volunteer->total_points
        ));
        
        if ($level) {
            $wpdb->update($table, array('level' => $level->level_number), array('id' => $id));
        }
    }
    
    public static function get_level_info($volunteer_id) {
        global $wpdb;
        $levels_table = Volunpress_DB::get_table('levels');
        $volunteer = self::get($volunteer_id);
        
        if (!$volunteer) return null;
        
        $current = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $levels_table WHERE level_number = %d",
            $volunteer->level
        ));
        
        $next = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $levels_table WHERE level_number = %d",
            $volunteer->level + 1
        ));
        
        return array(
            'current' => $current,
            'next' => $next,
            'points' => $volunteer->total_points,
            'progress' => $next ? min(100, round(($volunteer->total_points - $current->min_points) / ($next->min_points - $current->min_points) * 100)) : 100,
        );
    }
    
    public static function add_membership_points($id) {
        $points = (int) Volunpress::get_option('points_membership', 20);
        
        Volunpress_Activity::create(array(
            'volunteer_id' => $id,
            'activity_type' => 'membership',
            'description' => __('Cotisation annuelle', 'volunpress'),
            'points' => $points,
            'activity_date' => current_time('Y-m-d'),
        ));
    }
    
    public static function get_display_name($volunteer, $format = null) {
        if ($format === null) {
            $format = Volunpress::get_option('display_name_format', 'initial');
        }
        
        if ($format === 'full') {
            return esc_html($volunteer->first_name . ' ' . $volunteer->last_name);
        }
        
        return esc_html($volunteer->first_name . ' ' . strtoupper(substr($volunteer->last_name, 0, 1)) . '.');
    }
    
    public static function reactivate($id) {
        global $wpdb;
        $table = Volunpress_DB::get_table('volunteers');
        
        return $wpdb->update($table, array(
            'is_active' => 1,
            'left_date' => null,
        ), array('id' => $id));
    }
    
    public static function get_stats() {
        global $wpdb;
        $table = Volunpress_DB::get_table('volunteers');
        $activities_table = Volunpress_DB::get_table('activities');
        
        return array(
            'total_volunteers' => (int) $wpdb->get_var("SELECT COUNT(*) FROM $table"),
            'active_volunteers' => (int) $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE is_active = 1"),
            'committee_members' => (int) $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE is_committee_member = 1 AND is_active = 1"),
            'total_points_earned' => (int) $wpdb->get_var("SELECT COALESCE(SUM(points), 0) FROM $activities_table"),
            'total_hours' => (float) $wpdb->get_var("SELECT COALESCE(SUM(hours), 0) FROM $activities_table WHERE activity_type = 'hours'"),
        );
    }
}
