<?php
if (!defined('ABSPATH')) {
    exit;
}

class Volunpress_Shortcodes {
    
    public static function init() {
        add_shortcode('volunpress_leaderboard', array(__CLASS__, 'leaderboard'));
        add_shortcode('volunpress_volunteer', array(__CLASS__, 'volunteer_card'));
        add_shortcode('volunpress_rewards', array(__CLASS__, 'rewards_list'));
        add_shortcode('volunpress_stats', array(__CLASS__, 'stats_widget'));
    }
    
    // [volunpress_leaderboard limit="10" show_level="yes" show_badge="yes"]
    public static function leaderboard($atts) {
        $atts = shortcode_atts(array(
            'limit' => Volunpress::get_option('leaderboard_limit', 10),
            'show_level' => 'yes',
            'show_badge' => 'yes',
            'show_committee' => 'no',
            'committee_only' => 'no',
        ), $atts, 'volunpress_leaderboard');
        
        $args = array(
            'is_active' => 1,
            'orderby' => 'total_points',
            'order' => 'DESC',
            'limit' => (int) $atts['limit'],
        );
        
        if ($atts['committee_only'] === 'yes') {
            $args['is_committee_member'] = 1;
        }
        
        $volunteers = Volunpress_Volunteer::get_all($args);
        $currency = Volunpress::get_option('currency_name', 'points');
        $primary_color = Volunpress::get_option('primary_color', '#3498db');
        
        if (empty($volunteers)) {
            return '<p class="volunpress-empty">' . esc_html__('Aucun bénévole pour le moment.', 'volunpress') . '</p>';
        }
        
        ob_start();
        ?>
        <div class="volunpress-leaderboard" style="--vp-primary: <?php echo esc_attr($primary_color); ?>">
            <div class="vp-leaderboard-header">
                <span class="vp-rank-header">#</span>
                <span class="vp-name-header"><?php esc_html_e('Bénévole', 'volunpress'); ?></span>
                <span class="vp-points-header"><?php echo esc_html(ucfirst($currency)); ?></span>
            </div>
            <?php 
            $rank = 1;
            foreach ($volunteers as $volunteer): 
                $level_info = Volunpress_Volunteer::get_level_info($volunteer->id);
                $medal_class = $rank <= 3 ? 'vp-medal-' . $rank : '';
            ?>
            <div class="vp-leaderboard-row <?php echo esc_attr($medal_class); ?>" data-rank="<?php echo esc_attr($rank); ?>">
                <span class="vp-rank">
                    <?php if ($rank === 1): ?>
                        <span class="vp-medal">🥇</span>
                    <?php elseif ($rank === 2): ?>
                        <span class="vp-medal">🥈</span>
                    <?php elseif ($rank === 3): ?>
                        <span class="vp-medal">🥉</span>
                    <?php else: ?>
                        <?php echo esc_html($rank); ?>
                    <?php endif; ?>
                </span>
                <span class="vp-name">
                    <?php echo Volunpress_Volunteer::get_display_name($volunteer); ?>
                    <?php if ($atts['show_committee'] === 'yes' && $volunteer->is_committee_member): ?>
                        <span class="vp-committee-badge" title="<?php esc_attr_e('Membre du comité', 'volunpress'); ?>">⭐</span>
                    <?php endif; ?>
                    <?php if ($atts['show_level'] === 'yes' && $level_info): ?>
                        <span class="vp-level-badge" style="background-color: <?php echo esc_attr($level_info['current']->color); ?>">
                            <?php echo esc_html($level_info['current']->name); ?>
                        </span>
                    <?php endif; ?>
                </span>
                <span class="vp-points">
                    <strong><?php echo esc_html(number_format_i18n($volunteer->total_points)); ?></strong>
                    <?php if ($atts['show_badge'] === 'yes' && $level_info && $level_info['next']): ?>
                        <div class="vp-progress-mini">
                            <div class="vp-progress-bar" style="width: <?php echo esc_attr($level_info['progress']); ?>%"></div>
                        </div>
                    <?php endif; ?>
                </span>
            </div>
            <?php 
            $rank++;
            endforeach; 
            ?>
        </div>
        <?php
        return ob_get_clean();
    }
    
    // [volunpress_volunteer id="5"] ou [volunpress_volunteer email="john@example.com"]
    public static function volunteer_card($atts) {
        $atts = shortcode_atts(array(
            'id' => 0,
            'email' => '',
            'show_progress' => 'yes',
            'show_activities' => 'no',
            'activities_limit' => 5,
        ), $atts, 'volunpress_volunteer');
        
        $volunteer = null;
        
        if ($atts['id']) {
            $volunteer = Volunpress_Volunteer::get((int) $atts['id']);
        } elseif ($atts['email']) {
            $volunteer = Volunpress_Volunteer::get_by_email($atts['email']);
        }
        
        if (!$volunteer) {
            return '<p class="volunpress-empty">' . esc_html__('Bénévole non trouvé.', 'volunpress') . '</p>';
        }
        
        $level_info = Volunpress_Volunteer::get_level_info($volunteer->id);
        $currency = Volunpress::get_option('currency_name', 'points');
        $primary_color = Volunpress::get_option('primary_color', '#3498db');
        
        ob_start();
        ?>
        <div class="volunpress-card" style="--vp-primary: <?php echo esc_attr($primary_color); ?>; --vp-level-color: <?php echo esc_attr($level_info['current']->color ?? '#3498db'); ?>">
            <div class="vp-card-header">
                <div class="vp-avatar">
                    <?php echo esc_html(strtoupper(substr($volunteer->first_name, 0, 1) . substr($volunteer->last_name, 0, 1))); ?>
                </div>
                <div class="vp-card-info">
                    <h3 class="vp-card-name">
                        <?php echo Volunpress_Volunteer::get_display_name($volunteer); ?>
                        <?php if ($volunteer->is_committee_member): ?>
                            <span class="vp-committee-star">⭐</span>
                        <?php endif; ?>
                    </h3>
                    <span class="vp-card-level"><?php echo esc_html($level_info['current']->name ?? __('Débutant', 'volunpress')); ?></span>
                </div>
            </div>
            
            <div class="vp-card-points">
                <span class="vp-points-value"><?php echo esc_html(number_format_i18n($volunteer->total_points)); ?></span>
                <span class="vp-points-label"><?php echo esc_html($currency); ?></span>
            </div>
            
            <?php if ($atts['show_progress'] === 'yes' && $level_info['next']): ?>
            <div class="vp-card-progress">
                <div class="vp-progress-info">
                    <span><?php echo esc_html($level_info['current']->name); ?></span>
                    <span><?php echo esc_html($level_info['next']->name); ?></span>
                </div>
                <div class="vp-progress-track">
                    <div class="vp-progress-fill" style="width: <?php echo esc_attr($level_info['progress']); ?>%"></div>
                </div>
                <p class="vp-progress-text">
                    <?php 
                    $remaining = $level_info['next']->min_points - $volunteer->total_points;
                    printf(
                        esc_html__('Plus que %s %s pour le niveau suivant!', 'volunpress'),
                        '<strong>' . esc_html(number_format_i18n($remaining)) . '</strong>',
                        esc_html($currency)
                    ); 
                    ?>
                </p>
            </div>
            <?php endif; ?>
            
            <?php if ($atts['show_activities'] === 'yes'): ?>
            <div class="vp-card-activities">
                <h4><?php esc_html_e('Activités récentes', 'volunpress'); ?></h4>
                <?php 
                $activities = Volunpress_Activity::get_all(array(
                    'volunteer_id' => $volunteer->id,
                    'limit' => (int) $atts['activities_limit'],
                ));
                if ($activities): ?>
                <ul>
                    <?php foreach ($activities as $activity): ?>
                    <li>
                        <span class="vp-activity-desc"><?php echo esc_html($activity->description); ?></span>
                        <span class="vp-activity-points">+<?php echo esc_html($activity->points); ?></span>
                    </li>
                    <?php endforeach; ?>
                </ul>
                <?php else: ?>
                <p><?php esc_html_e('Aucune activité récente.', 'volunpress'); ?></p>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
    }
    
    // [volunpress_rewards]
    public static function rewards_list($atts) {
        $atts = shortcode_atts(array(
            'columns' => 3,
        ), $atts, 'volunpress_rewards');
        
        $rewards = Volunpress_Reward::get_all(array('is_active' => 1));
        $currency = Volunpress::get_option('currency_name', 'points');
        $primary_color = Volunpress::get_option('primary_color', '#3498db');
        
        if (empty($rewards)) {
            return '<p class="volunpress-empty">' . esc_html__('Aucune récompense disponible.', 'volunpress') . '</p>';
        }
        
        ob_start();
        ?>
        <div class="volunpress-rewards vp-cols-<?php echo esc_attr($atts['columns']); ?>" style="--vp-primary: <?php echo esc_attr($primary_color); ?>">
            <?php foreach ($rewards as $reward): ?>
            <div class="vp-reward-card <?php echo $reward->quantity_available === 0 ? 'vp-out-of-stock' : ''; ?>">
                <?php if ($reward->image_url): ?>
                <div class="vp-reward-image">
                    <img src="<?php echo esc_url($reward->image_url); ?>" alt="<?php echo esc_attr($reward->name); ?>">
                </div>
                <?php else: ?>
                <div class="vp-reward-image vp-reward-placeholder">🎁</div>
                <?php endif; ?>
                
                <div class="vp-reward-content">
                    <h4 class="vp-reward-name"><?php echo esc_html($reward->name); ?></h4>
                    <?php if ($reward->description): ?>
                    <p class="vp-reward-desc"><?php echo esc_html($reward->description); ?></p>
                    <?php endif; ?>
                    <div class="vp-reward-footer">
                        <span class="vp-reward-points"><?php echo esc_html(number_format_i18n($reward->points_required)); ?> <?php echo esc_html($currency); ?></span>
                        <?php if ($reward->quantity_available === 0): ?>
                        <span class="vp-reward-stock vp-stock-out"><?php esc_html_e('Épuisé', 'volunpress'); ?></span>
                        <?php elseif ($reward->quantity_available > 0): ?>
                        <span class="vp-reward-stock"><?php printf(esc_html__('%d disponible(s)', 'volunpress'), $reward->quantity_available); ?></span>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php
        return ob_get_clean();
    }
    
    // [volunpress_stats]
    public static function stats_widget($atts) {
        $atts = shortcode_atts(array(
            'show_volunteers' => 'yes',
            'show_hours' => 'yes',
            'show_points' => 'yes',
        ), $atts, 'volunpress_stats');
        
        $stats = Volunpress_Volunteer::get_stats();
        $currency = Volunpress::get_option('currency_name', 'points');
        $primary_color = Volunpress::get_option('primary_color', '#3498db');
        $secondary_color = Volunpress::get_option('secondary_color', '#2ecc71');
        
        ob_start();
        ?>
        <div class="volunpress-stats" style="--vp-primary: <?php echo esc_attr($primary_color); ?>; --vp-secondary: <?php echo esc_attr($secondary_color); ?>">
            <?php if ($atts['show_volunteers'] === 'yes'): ?>
            <div class="vp-stat-item">
                <span class="vp-stat-icon">👥</span>
                <span class="vp-stat-value"><?php echo esc_html(number_format_i18n($stats['active_volunteers'])); ?></span>
                <span class="vp-stat-label"><?php esc_html_e('Bénévoles actifs', 'volunpress'); ?></span>
            </div>
            <?php endif; ?>
            
            <?php if ($atts['show_hours'] === 'yes'): ?>
            <div class="vp-stat-item">
                <span class="vp-stat-icon">⏱️</span>
                <span class="vp-stat-value"><?php echo esc_html(number_format_i18n($stats['total_hours'], 1)); ?></span>
                <span class="vp-stat-label"><?php esc_html_e('Heures de bénévolat', 'volunpress'); ?></span>
            </div>
            <?php endif; ?>
            
            <?php if ($atts['show_points'] === 'yes'): ?>
            <div class="vp-stat-item">
                <span class="vp-stat-icon">⭐</span>
                <span class="vp-stat-value"><?php echo esc_html(number_format_i18n($stats['total_points_earned'])); ?></span>
                <span class="vp-stat-label"><?php echo esc_html(ucfirst($currency)); ?> <?php esc_html_e('distribués', 'volunpress'); ?></span>
            </div>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
    }
}
