<?php
if (!defined('ABSPATH')) {
    exit;
}

class Volunpress_Reward {
    
    public static function get_all($args = array()) {
        global $wpdb;
        $table = Volunpress_DB::get_table('rewards');
        
        $defaults = array(
            'is_active' => null,
            'orderby' => 'points_required',
            'order' => 'ASC',
        );
        $args = wp_parse_args($args, $defaults);
        
        $where = array('1=1');
        $values = array();
        
        if ($args['is_active'] !== null) {
            $where[] = 'is_active = %d';
            $values[] = (int) $args['is_active'];
        }
        
        $allowed_orderby = array('id', 'name', 'points_required', 'quantity_available');
        $orderby = in_array($args['orderby'], $allowed_orderby) ? $args['orderby'] : 'points_required';
        $order = strtoupper($args['order']) === 'DESC' ? 'DESC' : 'ASC';
        
        $sql = "SELECT * FROM $table WHERE " . implode(' AND ', $where) . " ORDER BY $orderby $order";
        
        if (!empty($values)) {
            $sql = $wpdb->prepare($sql, $values);
        }
        
        return $wpdb->get_results($sql);
    }
    
    public static function get($id) {
        global $wpdb;
        $table = Volunpress_DB::get_table('rewards');
        return $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $id));
    }
    
    public static function create($data) {
        global $wpdb;
        $table = Volunpress_DB::get_table('rewards');
        
        $insert_data = array(
            'name' => sanitize_text_field($data['name']),
            'description' => isset($data['description']) ? sanitize_textarea_field($data['description']) : '',
            'points_required' => (int) $data['points_required'],
            'quantity_available' => isset($data['quantity_available']) ? (int) $data['quantity_available'] : -1,
            'image_url' => isset($data['image_url']) ? esc_url_raw($data['image_url']) : '',
            'is_active' => isset($data['is_active']) ? (int) $data['is_active'] : 1,
        );
        
        $result = $wpdb->insert($table, $insert_data);
        return $result ? $wpdb->insert_id : false;
    }
    
    public static function update($id, $data) {
        global $wpdb;
        $table = Volunpress_DB::get_table('rewards');
        
        $update_data = array();
        
        if (isset($data['name'])) {
            $update_data['name'] = sanitize_text_field($data['name']);
        }
        if (isset($data['description'])) {
            $update_data['description'] = sanitize_textarea_field($data['description']);
        }
        if (isset($data['points_required'])) {
            $update_data['points_required'] = (int) $data['points_required'];
        }
        if (isset($data['quantity_available'])) {
            $update_data['quantity_available'] = (int) $data['quantity_available'];
        }
        if (isset($data['image_url'])) {
            $update_data['image_url'] = esc_url_raw($data['image_url']);
        }
        if (isset($data['is_active'])) {
            $update_data['is_active'] = (int) $data['is_active'];
        }
        
        if (empty($update_data)) return true;
        
        return $wpdb->update($table, $update_data, array('id' => $id));
    }
    
    public static function delete($id) {
        global $wpdb;
        $table = Volunpress_DB::get_table('rewards');
        return $wpdb->delete($table, array('id' => $id));
    }
    
    public static function claim($volunteer_id, $reward_id) {
        global $wpdb;
        $claims_table = Volunpress_DB::get_table('claims');
        $rewards_table = Volunpress_DB::get_table('rewards');
        
        $volunteer = Volunpress_Volunteer::get($volunteer_id);
        $reward = self::get($reward_id);
        
        if (!$volunteer || !$reward) {
            return new WP_Error('not_found', __('Bénévole ou récompense introuvable.', 'volunpress'));
        }
        
        if (!$reward->is_active) {
            return new WP_Error('inactive', __('Cette récompense n\'est plus disponible.', 'volunpress'));
        }
        
        if ($reward->quantity_available === 0) {
            return new WP_Error('out_of_stock', __('Cette récompense est en rupture de stock.', 'volunpress'));
        }
        
        if ($volunteer->total_points < $reward->points_required) {
            return new WP_Error('insufficient_points', __('Points insuffisants pour cette récompense.', 'volunpress'));
        }
        
        $result = $wpdb->insert($claims_table, array(
            'volunteer_id' => $volunteer_id,
            'reward_id' => $reward_id,
            'points_spent' => $reward->points_required,
            'status' => 'pending',
        ));
        
        if ($result) {
            // Déduire les points
            Volunpress_Volunteer::deduct_points($volunteer_id, $reward->points_required);
            
            // Diminuer la quantité si limitée
            if ($reward->quantity_available > 0) {
                $wpdb->query($wpdb->prepare(
                    "UPDATE $rewards_table SET quantity_available = quantity_available - 1 WHERE id = %d",
                    $reward_id
                ));
            }
            
            return $wpdb->insert_id;
        }
        
        return new WP_Error('error', __('Erreur lors de la réclamation.', 'volunpress'));
    }
    
    public static function get_claims($args = array()) {
        global $wpdb;
        $claims_table = Volunpress_DB::get_table('claims');
        $volunteers_table = Volunpress_DB::get_table('volunteers');
        $rewards_table = Volunpress_DB::get_table('rewards');
        
        $defaults = array(
            'volunteer_id' => null,
            'status' => null,
            'orderby' => 'claimed_at',
            'order' => 'DESC',
        );
        $args = wp_parse_args($args, $defaults);
        
        $where = array('1=1');
        $values = array();
        
        if ($args['volunteer_id']) {
            $where[] = 'c.volunteer_id = %d';
            $values[] = (int) $args['volunteer_id'];
        }
        
        if ($args['status']) {
            $where[] = 'c.status = %s';
            $values[] = sanitize_text_field($args['status']);
        }
        
        $sql = "SELECT c.*, v.first_name, v.last_name, r.name as reward_name 
                FROM $claims_table c 
                LEFT JOIN $volunteers_table v ON c.volunteer_id = v.id 
                LEFT JOIN $rewards_table r ON c.reward_id = r.id 
                WHERE " . implode(' AND ', $where) . " 
                ORDER BY c.claimed_at DESC";
        
        if (!empty($values)) {
            $sql = $wpdb->prepare($sql, $values);
        }
        
        return $wpdb->get_results($sql);
    }
    
    public static function update_claim_status($claim_id, $status) {
        global $wpdb;
        $claims_table = Volunpress_DB::get_table('claims');
        
        $allowed_statuses = array('pending', 'approved', 'rejected', 'delivered');
        if (!in_array($status, $allowed_statuses)) {
            return false;
        }
        
        $update_data = array('status' => $status);
        
        if ($status !== 'pending') {
            $update_data['processed_at'] = current_time('mysql');
            $update_data['processed_by'] = get_current_user_id();
        }
        
        // Si rejeté, rembourser les points
        if ($status === 'rejected') {
            $claim = $wpdb->get_row($wpdb->prepare("SELECT * FROM $claims_table WHERE id = %d", $claim_id));
            if ($claim && $claim->status === 'pending') {
                Volunpress_Volunteer::add_points($claim->volunteer_id, $claim->points_spent);
            }
        }
        
        return $wpdb->update($claims_table, $update_data, array('id' => $claim_id));
    }
    
    public static function get_available_for_volunteer($volunteer_id) {
        $volunteer = Volunpress_Volunteer::get($volunteer_id);
        if (!$volunteer) return array();
        
        $rewards = self::get_all(array('is_active' => 1));
        $available = array();
        
        foreach ($rewards as $reward) {
            $reward->can_claim = ($volunteer->total_points >= $reward->points_required && ($reward->quantity_available === -1 || $reward->quantity_available > 0));
            $available[] = $reward;
        }
        
        return $available;
    }
}
