<?php
if (!defined('ABSPATH')) {
    exit;
}

class Volunpress_Activity {
    
    public static function get_all($args = array()) {
        global $wpdb;
        $table = Volunpress_DB::get_table('activities');
        $volunteers_table = Volunpress_DB::get_table('volunteers');
        
        $defaults = array(
            'volunteer_id' => null,
            'activity_type' => null,
            'date_from' => null,
            'date_to' => null,
            'orderby' => 'activity_date',
            'order' => 'DESC',
            'limit' => -1,
            'offset' => 0,
        );
        $args = wp_parse_args($args, $defaults);
        
        $where = array('1=1');
        $values = array();
        
        if ($args['volunteer_id']) {
            $where[] = 'a.volunteer_id = %d';
            $values[] = (int) $args['volunteer_id'];
        }
        
        if ($args['activity_type']) {
            $where[] = 'a.activity_type = %s';
            $values[] = sanitize_text_field($args['activity_type']);
        }
        
        if ($args['date_from']) {
            $where[] = 'a.activity_date >= %s';
            $values[] = sanitize_text_field($args['date_from']);
        }
        
        if ($args['date_to']) {
            $where[] = 'a.activity_date <= %s';
            $values[] = sanitize_text_field($args['date_to']);
        }
        
        $allowed_orderby = array('id', 'activity_date', 'points', 'hours', 'activity_type');
        $orderby = in_array($args['orderby'], $allowed_orderby) ? 'a.' . $args['orderby'] : 'a.activity_date';
        $order = strtoupper($args['order']) === 'ASC' ? 'ASC' : 'DESC';
        
        $sql = "SELECT a.*, v.first_name, v.last_name 
                FROM $table a 
                LEFT JOIN $volunteers_table v ON a.volunteer_id = v.id 
                WHERE " . implode(' AND ', $where) . " 
                ORDER BY $orderby $order";
        
        if ($args['limit'] > 0) {
            $sql .= " LIMIT " . intval($args['limit']) . " OFFSET " . intval($args['offset']);
        }
        
        if (!empty($values)) {
            $sql = $wpdb->prepare($sql, $values);
        }
        
        return $wpdb->get_results($sql);
    }
    
    public static function get($id) {
        global $wpdb;
        $table = Volunpress_DB::get_table('activities');
        return $wpdb->get_row($wpdb->prepare("SELECT * FROM $table WHERE id = %d", $id));
    }
    
    public static function create($data) {
        global $wpdb;
        $table = Volunpress_DB::get_table('activities');
        
        $activity_type = sanitize_text_field($data['activity_type']);
        $hours = isset($data['hours']) ? floatval($data['hours']) : 0;
        
        // Calculer les points si c'est des heures
        if ($activity_type === 'hours' && isset($data['hours'])) {
            $points_per_hour = (int) Volunpress::get_option('points_per_hour', 10);
            $points = round($hours * $points_per_hour);
        } else {
            $points = isset($data['points']) ? (int) $data['points'] : 0;
        }
        
        $insert_data = array(
            'volunteer_id' => (int) $data['volunteer_id'],
            'activity_type' => $activity_type,
            'description' => sanitize_text_field($data['description']),
            'hours' => $hours,
            'points' => $points,
            'activity_date' => sanitize_text_field($data['activity_date']),
            'added_by' => get_current_user_id(),
        );
        
        $result = $wpdb->insert($table, $insert_data);
        
        if ($result) {
            Volunpress_Volunteer::add_points($data['volunteer_id'], $points);
            return $wpdb->insert_id;
        }
        
        return false;
    }
    
    public static function update($id, $data) {
        global $wpdb;
        $table = Volunpress_DB::get_table('activities');
        
        $activity = self::get($id);
        if (!$activity) return false;
        
        $old_points = $activity->points;
        $update_data = array();
        
        if (isset($data['description'])) {
            $update_data['description'] = sanitize_text_field($data['description']);
        }
        
        if (isset($data['activity_date'])) {
            $update_data['activity_date'] = sanitize_text_field($data['activity_date']);
        }
        
        if (isset($data['hours'])) {
            $update_data['hours'] = floatval($data['hours']);
            if ($activity->activity_type === 'hours') {
                $points_per_hour = (int) Volunpress::get_option('points_per_hour', 10);
                $update_data['points'] = round($update_data['hours'] * $points_per_hour);
            }
        }
        
        if (isset($data['points']) && $activity->activity_type !== 'hours') {
            $update_data['points'] = (int) $data['points'];
        }
        
        if (empty($update_data)) return true;
        
        $result = $wpdb->update($table, $update_data, array('id' => $id));
        
        // Recalculer les points du bénévole
        if ($result && isset($update_data['points'])) {
            Volunpress_Volunteer::recalculate_points($activity->volunteer_id);
        }
        
        return $result;
    }
    
    public static function delete($id) {
        global $wpdb;
        $table = Volunpress_DB::get_table('activities');
        
        $activity = self::get($id);
        if (!$activity) return false;
        
        $result = $wpdb->delete($table, array('id' => $id));
        
        if ($result) {
            Volunpress_Volunteer::recalculate_points($activity->volunteer_id);
        }
        
        return $result;
    }
    
    public static function get_activity_types() {
        return array(
            'hours' => __('Heures de bénévolat', 'volunpress'),
            'donation' => __('Don', 'volunpress'),
            'membership' => __('Cotisation', 'volunpress'),
            'bonus' => __('Bonus', 'volunpress'),
            'event' => __('Événement', 'volunpress'),
            'other' => __('Autre', 'volunpress'),
        );
    }
    
    public static function get_recent($limit = 10) {
        return self::get_all(array('limit' => $limit, 'orderby' => 'created_at', 'order' => 'DESC'));
    }
    
    public static function get_monthly_stats($year = null, $month = null) {
        global $wpdb;
        $table = Volunpress_DB::get_table('activities');
        
        if (!$year) $year = date('Y');
        if (!$month) $month = date('m');
        
        $start = "$year-$month-01";
        $end = date('Y-m-t', strtotime($start));
        
        return array(
            'total_points' => (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COALESCE(SUM(points), 0) FROM $table WHERE activity_date BETWEEN %s AND %s",
                $start, $end
            )),
            'total_hours' => (float) $wpdb->get_var($wpdb->prepare(
                "SELECT COALESCE(SUM(hours), 0) FROM $table WHERE activity_type = 'hours' AND activity_date BETWEEN %s AND %s",
                $start, $end
            )),
            'total_activities' => (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table WHERE activity_date BETWEEN %s AND %s",
                $start, $end
            )),
        );
    }
}
