<?php
if (!defined('ABSPATH')) {
    exit;
}

class Volunpress_Admin {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('admin_init', array($this, 'handle_actions'));
        add_action('wp_ajax_volunpress_quick_add_points', array($this, 'ajax_quick_add_points'));
    }
    
    public function add_menu() {
        add_menu_page(
            __('Volunpress', 'volunpress'),
            __('Volunpress', 'volunpress'),
            'manage_options',
            'volunpress',
            array($this, 'page_dashboard'),
            'dashicons-groups',
            30
        );
        
        add_submenu_page('volunpress', __('Tableau de bord', 'volunpress'), __('Tableau de bord', 'volunpress'), 'manage_options', 'volunpress', array($this, 'page_dashboard'));
        add_submenu_page('volunpress', __('Bénévoles', 'volunpress'), __('Bénévoles', 'volunpress'), 'manage_options', 'volunpress-volunteers', array($this, 'page_volunteers'));
        add_submenu_page('volunpress', __('Activités', 'volunpress'), __('Activités', 'volunpress'), 'manage_options', 'volunpress-activities', array($this, 'page_activities'));
        add_submenu_page('volunpress', __('Récompenses', 'volunpress'), __('Récompenses', 'volunpress'), 'manage_options', 'volunpress-rewards', array($this, 'page_rewards'));
        add_submenu_page('volunpress', __('Réclamations', 'volunpress'), __('Réclamations', 'volunpress'), 'manage_options', 'volunpress-claims', array($this, 'page_claims'));
        add_submenu_page('volunpress', __('Options', 'volunpress'), __('Options', 'volunpress'), 'manage_options', 'volunpress-options', array($this, 'page_options'));
    }
    
    public function enqueue_assets($hook) {
        if (strpos($hook, 'volunpress') === false) return;
        
        wp_enqueue_style('volunpress-admin', VOLUNPRESS_PLUGIN_URL . 'assets/css/admin.css', array(), VOLUNPRESS_VERSION);
        wp_enqueue_script('volunpress-admin', VOLUNPRESS_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), VOLUNPRESS_VERSION, true);
        wp_enqueue_media();
        
        wp_localize_script('volunpress-admin', 'volunpress', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('volunpress_admin'),
        ));
    }
    
    public function handle_actions() {
        if (!current_user_can('manage_options')) return;
        
        // Gestion des bénévoles
        if (isset($_POST['volunpress_save_volunteer']) && wp_verify_nonce($_POST['_wpnonce'], 'volunpress_volunteer')) {
            $id = isset($_POST['volunteer_id']) ? (int) $_POST['volunteer_id'] : 0;
            $data = array(
                'first_name' => $_POST['first_name'],
                'last_name' => $_POST['last_name'],
                'email' => $_POST['email'],
                'phone' => $_POST['phone'] ?? '',
                'is_committee_member' => isset($_POST['is_committee_member']) ? 1 : 0,
                'is_active' => isset($_POST['is_active']) ? 1 : 0,
                'membership_paid' => isset($_POST['membership_paid']) ? 1 : 0,
                'membership_year' => $_POST['membership_year'] ?? null,
                'joined_date' => $_POST['joined_date'],
                'left_date' => $_POST['left_date'] ?? null,
                'notes' => $_POST['notes'] ?? '',
            );
            
            if ($id) {
                Volunpress_Volunteer::update($id, $data);
                $message = __('Bénévole mis à jour.', 'volunpress');
            } else {
                Volunpress_Volunteer::create($data);
                $message = __('Bénévole ajouté.', 'volunpress');
            }
            
            wp_redirect(admin_url('admin.php?page=volunpress-volunteers&message=' . urlencode($message)));
            exit;
        }
        
        // Suppression bénévole
        if (isset($_GET['action']) && $_GET['action'] === 'delete_volunteer' && isset($_GET['id']) && wp_verify_nonce($_GET['_wpnonce'], 'delete_volunteer_' . $_GET['id'])) {
            Volunpress_Volunteer::delete((int) $_GET['id']);
            wp_redirect(admin_url('admin.php?page=volunpress-volunteers&message=' . urlencode(__('Bénévole supprimé.', 'volunpress'))));
            exit;
        }
        
        // Gestion des activités
        if (isset($_POST['volunpress_save_activity']) && wp_verify_nonce($_POST['_wpnonce'], 'volunpress_activity')) {
            Volunpress_Activity::create(array(
                'volunteer_id' => $_POST['volunteer_id'],
                'activity_type' => $_POST['activity_type'],
                'description' => $_POST['description'],
                'hours' => $_POST['hours'] ?? 0,
                'points' => $_POST['points'] ?? 0,
                'activity_date' => $_POST['activity_date'],
            ));
            
            wp_redirect(admin_url('admin.php?page=volunpress-activities&message=' . urlencode(__('Activité ajoutée.', 'volunpress'))));
            exit;
        }
        
        // Suppression activité
        if (isset($_GET['action']) && $_GET['action'] === 'delete_activity' && isset($_GET['id']) && wp_verify_nonce($_GET['_wpnonce'], 'delete_activity_' . $_GET['id'])) {
            Volunpress_Activity::delete((int) $_GET['id']);
            wp_redirect(admin_url('admin.php?page=volunpress-activities&message=' . urlencode(__('Activité supprimée.', 'volunpress'))));
            exit;
        }
        
        // Gestion des récompenses
        if (isset($_POST['volunpress_save_reward']) && wp_verify_nonce($_POST['_wpnonce'], 'volunpress_reward')) {
            $id = isset($_POST['reward_id']) ? (int) $_POST['reward_id'] : 0;
            $data = array(
                'name' => $_POST['name'],
                'description' => $_POST['description'] ?? '',
                'points_required' => $_POST['points_required'],
                'quantity_available' => $_POST['quantity_available'] ?? -1,
                'image_url' => $_POST['image_url'] ?? '',
                'is_active' => isset($_POST['is_active']) ? 1 : 0,
            );
            
            if ($id) {
                Volunpress_Reward::update($id, $data);
            } else {
                Volunpress_Reward::create($data);
            }
            
            wp_redirect(admin_url('admin.php?page=volunpress-rewards&message=' . urlencode(__('Récompense enregistrée.', 'volunpress'))));
            exit;
        }
        
        // Suppression récompense
        if (isset($_GET['action']) && $_GET['action'] === 'delete_reward' && isset($_GET['id']) && wp_verify_nonce($_GET['_wpnonce'], 'delete_reward_' . $_GET['id'])) {
            Volunpress_Reward::delete((int) $_GET['id']);
            wp_redirect(admin_url('admin.php?page=volunpress-rewards&message=' . urlencode(__('Récompense supprimée.', 'volunpress'))));
            exit;
        }
        
        // Mise à jour statut réclamation
        if (isset($_GET['action']) && in_array($_GET['action'], array('approve_claim', 'reject_claim', 'deliver_claim')) && isset($_GET['id']) && wp_verify_nonce($_GET['_wpnonce'], 'claim_action_' . $_GET['id'])) {
            $status_map = array('approve_claim' => 'approved', 'reject_claim' => 'rejected', 'deliver_claim' => 'delivered');
            Volunpress_Reward::update_claim_status((int) $_GET['id'], $status_map[$_GET['action']]);
            wp_redirect(admin_url('admin.php?page=volunpress-claims&message=' . urlencode(__('Statut mis à jour.', 'volunpress'))));
            exit;
        }
        
        // Options
        if (isset($_POST['volunpress_save_options']) && wp_verify_nonce($_POST['_wpnonce'], 'volunpress_options')) {
            $options = array(
                'points_per_hour' => (int) $_POST['points_per_hour'],
                'points_membership' => (int) $_POST['points_membership'],
                'display_name_format' => $_POST['display_name_format'],
                'currency_name' => sanitize_text_field($_POST['currency_name']),
                'organization_name' => sanitize_text_field($_POST['organization_name']),
                'primary_color' => sanitize_hex_color($_POST['primary_color']),
                'secondary_color' => sanitize_hex_color($_POST['secondary_color']),
                'show_leaderboard' => isset($_POST['show_leaderboard']) ? 1 : 0,
                'leaderboard_limit' => (int) $_POST['leaderboard_limit'],
                'donation_points_ratio' => (float) $_POST['donation_points_ratio'],
            );
            
            update_option('volunpress_options', $options);
            wp_redirect(admin_url('admin.php?page=volunpress-options&message=' . urlencode(__('Options enregistrées.', 'volunpress'))));
            exit;
        }
    }
    
    public function ajax_quick_add_points() {
        check_ajax_referer('volunpress_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $volunteer_id = (int) $_POST['volunteer_id'];
        $points = (int) $_POST['points'];
        $description = sanitize_text_field($_POST['description']);
        
        Volunpress_Activity::create(array(
            'volunteer_id' => $volunteer_id,
            'activity_type' => 'bonus',
            'description' => $description,
            'points' => $points,
            'activity_date' => current_time('Y-m-d'),
        ));
        
        $volunteer = Volunpress_Volunteer::get($volunteer_id);
        wp_send_json_success(array('new_total' => $volunteer->total_points));
    }
    
    // Pages d'administration
    public function page_dashboard() {
        require_once VOLUNPRESS_PLUGIN_DIR . 'admin/views/dashboard.php';
    }
    
    public function page_volunteers() {
        require_once VOLUNPRESS_PLUGIN_DIR . 'admin/views/volunteers.php';
    }
    
    public function page_activities() {
        require_once VOLUNPRESS_PLUGIN_DIR . 'admin/views/activities.php';
    }
    
    public function page_rewards() {
        require_once VOLUNPRESS_PLUGIN_DIR . 'admin/views/rewards.php';
    }
    
    public function page_claims() {
        require_once VOLUNPRESS_PLUGIN_DIR . 'admin/views/claims.php';
    }
    
    public function page_options() {
        require_once VOLUNPRESS_PLUGIN_DIR . 'admin/views/options.php';
    }
}

new Volunpress_Admin();
